<?php

namespace App\Http\Controllers;

use App\Models\Society;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

use App\Services\SocietyService;

class SocietyController extends Controller
{
    protected $societyService;

    protected $superAdminService;

    public function __construct(SocietyService $societyService, \App\Services\SuperAdminService $superAdminService)
    {
        $this->societyService = $societyService;
        $this->superAdminService = $superAdminService;
    }

    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:active,suspended'
        ]);

        // Authorization check: Ensure only super admin can do this
        // Ideally handled by middleware, but double check here if needed
        // Assuming route middleware 'can:super_admin' or similar

        $society = $this->superAdminService->updateSocietyStatus($id, $request->status);

        return $this->success($society, 'Society status updated successfully');
    }

    public function index()
    {
        // For Super Admin: List all societies with pagination
        return response()->json(Society::paginate(10));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'registration_number' => 'nullable|string|unique:societies',
            'society_photo' => 'nullable|string',
            'address' => 'nullable|string',
            'area' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'pincode' => 'nullable|string',
            'logo' => 'nullable|string',
            'contact_no' => 'nullable|string',
            'email' => 'nullable|email',
        ]);

        $society = $this->societyService->createSociety($validated);

        return response()->json([
            'status' => 'success',
            'message' => 'Society created successfully',
            'data' => $society
        ], 201);
    }

    public function show($id)
    {
        $society = Society::with('users')->findOrFail($id);
        return response()->json($society);
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'registration_number' => 'nullable|string|unique:societies,registration_number,' . $id,
            'society_photo' => 'nullable|string',
            'address' => 'nullable|string',
            'area' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'pincode' => 'nullable|string',
            'logo' => 'nullable|string',
            'contact_no' => 'nullable|string',
            'email' => 'nullable|email',
            'configuration' => 'nullable|string',
        ]);

        $society = $this->societyService->updateSociety($id, $validated);

        return response()->json([
            'status' => 'success',
            'message' => 'Society updated successfully',
            'data' => $society
        ]);
    }

    public function destroy($id)
    {
        $this->societyService->deleteSociety($id);

        return response()->json([
            'status' => 'success',
            'message' => 'Society deleted successfully'
        ]);
    }

    // New: Manage Society Users (for Society Admin) with pagination
    public function getUsers($societyId)
    {
        $users = User::where('society_id', $societyId)->paginate(10);
        return response()->json($users);
    }

    public function addUser(Request $request, $societyId)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'role' => 'required|string', // member, secretary, vendor, employee
            'password' => 'required|string|min:6',

            // Member fields
            'house_number' => 'nullable|string',
            'wing' => 'nullable|string',
            'member_type' => 'nullable|string',

            // Employee fields
            'employee_id_code' => 'nullable|string',
            'designation' => 'nullable|string',
            'joining_date' => 'nullable|date',

            // Vendor fields
            'vendor_id_code' => 'nullable|string',
            'vendor_type' => 'nullable|string',
            'service_description' => 'nullable|string',

            'phone' => 'nullable|string',
            'photo' => 'nullable|string',
        ]);

        $user = $this->societyService->addUserToSociety($societyId, $validated);

        return response()->json([
            'status' => 'success',
            'message' => 'User added to society successfully',
            'data' => $user
        ], 201);
    }

    public function deleteUser($societyId, $userId)
    {
        $this->societyService->deleteUserFromSociety($societyId, $userId);

        return response()->json([
            'status' => 'success',
            'message' => 'User removed from society successfully'
        ]);
    }

    public function updateUser(Request $request, $societyId, $userId)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $userId,
            'role' => 'required|string',
            'password' => 'nullable|string|min:6',

            // Member fields
            'house_number' => 'nullable|string',
            'wing' => 'nullable|string',
            'member_type' => 'nullable|string',

            // Employee fields
            'employee_id_code' => 'nullable|string',
            'designation' => 'nullable|string',
            'joining_date' => 'nullable|date',

            // Vendor fields
            'vendor_id_code' => 'nullable|string',
            'vendor_type' => 'nullable|string',
            'service_description' => 'nullable|string',

            'phone' => 'nullable|string',
            'photo' => 'nullable|string',
        ]);

        $user = $this->societyService->updateUserInSociety($societyId, $userId, $validated);

        return response()->json([
            'status' => 'success',
            'message' => 'User updated successfully',
            'data' => $user
        ]);
    }
}
