<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;

use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    public function login(Request $request)
    {
        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            $user = User::with('society')->find(Auth::id());
            $token = \Illuminate\Support\Str::random(60);
            $user->api_token = $token;
            $user->save();

            $rolePermissions = [
                'super_admin' => ['societies', 'users'],
                'society_admin' => ['society_profile', 'committee', 'members', 'vendors', 'employees', 'tenders', 'finance', 'units', 'maintenance'],
                'member' => ['my_society'],
                'secretary' => ['society_profile', 'committee', 'members', 'tenders', 'finance', 'units', 'maintenance'],
                'vendor' => ['my_tasks'],
                'employee' => ['my_tasks'],
            ];

            return response()->json([
                'status' => 'success',
                'user' => [
                    'name' => $user->name,
                    'email' => $user->email,
                    'role' => $user->role,
                    'society' => $user->society,
                    'permissions' => $rolePermissions[$user->role] ?? []
                ],
                'token' => $token
            ]);
        }

        return response()->json([
            'status' => 'error',
            'message' => 'Invalid credentials'
        ], 401);
    }

    public function me(Request $request)
    {
        $user = Auth::user();
        if ($user) {
            return response()->json([
                'name' => $user->name,
                'role' => $user->role,
            ]);
        }
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    public function changePassword(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'current_password' => 'required',
            'new_password' => 'required|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first()
            ], 422);
        }

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Current password is incorrect'
            ], 400);
        }

        $user->password = Hash::make($request->new_password);
        $user->save();

        return response()->json([
            'status' => 'success',
            'message' => 'Password updated successfully'
        ]);
    }

    public function logout(Request $request)
    {
        Auth::logout();
        return response()->json(['message' => 'Logged out']);
    }
}
